/* vi: set sw=4 ts=4: */
/*
 * RFC1035 domain compression routines (C) 2007 Gabriel Somlo <somlo at cmu.edu>
 *
 * Loosely based on the isc-dhcpd implementation by dhankins@isc.org
 *
 * Licensed under GPLv2 or later, see file LICENSE in this source tree.
 */
#include "common.h"
#define NS_MAXDNAME 1025
#define NS_MAXCDNAME 255
#define NS_MAXLABEL 63
#define NS_MAXDNSRCH 6
#define NS_CMPRSFLGS 0xc0
/* Expand a RFC1035-compressed list of domain names "cstr", of length "clen";
 * return a newly allocated string containing the space-separated domains,
 * prefixed with the contents of string pre, or NULL if an error occurs.
 */
char* FAST_FUNC dname_dec(const uint8_t *cstr, int clen, const char *pre)
{
	char *ret, *end;
	unsigned len, crtpos, retpos, depth;
	crtpos = retpos = depth = 0;
	len = strlen(pre);
	end = ret = xstrdup(pre);
	/* Scan the string once, allocating new memory as needed */
	while (crtpos < clen) {
		const uint8_t *c;
		c = cstr + crtpos;
		if ((*c & NS_CMPRSFLGS) == NS_CMPRSFLGS) {
			/* pointer */
			if (crtpos + 2 > clen) /* no offset to jump to? abort */
				goto error;
			if (retpos == 0) /* toplevel? save return spot */
				retpos = crtpos + 2;
			depth++;
			crtpos = ((c[0] << 8) | c[1]) & 0x3fff; /* jump */
		} else if (*c) {
			unsigned label_len;
			/* label */
			if (crtpos + *c + 1 > clen) /* label too long? abort */
				goto error;
			ret = xrealloc(ret, len + *c + 1);
			/* \3com ---> "com." */
			end = (char *)mempcpy(ret + len, c + 1, *c);
			*end = '.';
			label_len = *c + 1;
			len += label_len;
			crtpos += label_len;
		} else {
			/* NUL: end of current domain name */
			if (retpos == 0) {
				/* toplevel? keep going */
				crtpos++;
			} else {
				/* return to toplevel saved spot */
				crtpos = retpos;
				retpos = depth = 0;
			}
			if (len != 0) {
				/* \4host\3com\0\4host and we are at \0:
				 * \3com was converted to "com.", change dot to space.
				 */
				ret[len - 1] = ' ';
			}
		}
		if (depth > NS_MAXDNSRCH /* too many jumps? abort, it's a loop */
		 || len > NS_MAXDNAME * NS_MAXDNSRCH /* result too long? abort */
		) {
			goto error;
		}
	}
	if (ret == end) { /* expanded string is empty? abort */
 error:
		free(ret);
		return NULL;
	}
	*end = '\0';
	return ret;
}
/* Convert a domain name (src) from human-readable "foo.BLAH.com" format into
 * RFC1035 encoding "\003foo\004blah\003com\000". Return allocated string, or
 * NULL if an error occurs.
 */
static uint8_t *convert_dname(const char *src, int *retlen)
{
	uint8_t *res, *lenptr, *dst;
	res = xzalloc(strlen(src) + 2);
	dst = lenptr = res;
	dst++;
	for (;;) {
		uint8_t c;
		int len;
		c = (uint8_t)*src++;
		if (c == '.' || c == '\0') {  /* end of label */
			len = dst - lenptr - 1;
			/* label too long, too short, or two '.'s in a row (len will be 0) */
			if (len > NS_MAXLABEL || len == 0)
				goto error;
			*lenptr = len;
			if (c == '\0' || *src == '\0')	/* "" or ".": end of src */
				break;
			lenptr = dst++;
			continue;
		}
		*dst++ = tolower(c);
	}
	*retlen = dst + 1 - res;
	if (*retlen > NS_MAXCDNAME) {  /* dname too long? abort */
 error:
		free(res);
		*retlen = 0;
		return NULL;
	}
	return res;
}
uint8_t* FAST_FUNC dname_enc(/*const uint8_t *cstr, int clen,*/ const char *src, int *retlen)
{
	return convert_dname(src, retlen);
}
