/* vi: set sw=4 ts=4: */
/*
 * Rexec program for system have fork() as vfork() with foreground option
 *
 * Copyright (C) Vladimir N. Oleynik <dzo@simtreas.ru>
 * Copyright (C) 2003 Russ Dill <Russ.Dill@asu.edu>
 *
 * daemon() portion taken from uClibc:
 *
 * Copyright (c) 1991, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * Modified for uClibc by Erik Andersen <andersee@debian.org>
 *
 * Licensed under GPLv2 or later, see file LICENSE in this source tree.
 */
#include "busybox.h"
#include "NUM_APPLETS.h"
#define NOFORK_SUPPORT ((NUM_APPLETS > 1) && (ENABLE_FEATURE_PREFER_APPLETS || ENABLE_FEATURE_SH_NOFORK))
#define NOEXEC_SUPPORT ((NUM_APPLETS > 1) && (ENABLE_FEATURE_PREFER_APPLETS || ENABLE_FEATURE_SH_STANDALONE))
#include <sys/prctl.h>
void FAST_FUNC set_task_comm(const char *comm)
{
	/* okay if too long (truncates) */
	prctl(PR_SET_NAME, (long)comm, 0, 0, 0);
}
/*
 * NOFORK/NOEXEC support
 */
/*
 * Higher-level code, hiding optional NOFORK/NOEXEC trickery.
 */
/* This does a fork/exec in one call, using vfork().  Returns PID of new child,
 * -1 for failure.  Runs argv[0], searching path if that has no / in it. */
pid_t FAST_FUNC spawn(char **argv)
{
	/* Compiler should not optimize stores here */
	volatile int failed;
	pid_t pid;
	fflush_all();
	/* Be nice to nommu machines. */
	failed = 0;
	pid = vfork();
	if (pid < 0) /* error */
		return pid;
	if (!pid) { /* child */
		/* This macro is ok - it doesn't do NOEXEC/NOFORK tricks */
		BB_EXECVP(argv[0], argv);
		/* We are (maybe) sharing a stack with blocked parent,
		 * let parent know we failed and then exit to unblock parent
		 * (but don't run atexit() stuff, which would screw up parent.)
		 */
		failed = errno;
		/* mount, for example, does not want the message */
		/*bb_perror_msg("can't execute '%s'", argv[0]);*/
		_exit(111);
	}
	/* parent */
	/* Unfortunately, this is not reliable: according to standards
	 * vfork() can be equivalent to fork() and we won't see value
	 * of 'failed'.
	 * Interested party can wait on pid and learn exit code.
	 * If 111 - then it (most probably) failed to exec */
	if (failed) {
		safe_waitpid(pid, NULL, 0); /* prevent zombie */
		errno = failed;
		return -1;
	}
	return pid;
}
/* Die with an error message if we can't spawn a child process. */
pid_t FAST_FUNC xspawn(char **argv)
{
	pid_t pid = spawn(argv);
	if (pid < 0)
		bb_simple_perror_msg_and_die(*argv);
	return pid;
}
int FAST_FUNC spawn_and_wait(char **argv)
{
	int rc;
	rc = spawn(argv);
	return wait4pid(rc);
}
/* Due to a #define in libbb.h on MMU systems we actually have 1 argument -
 * char **argv "vanishes" */
void FAST_FUNC bb_daemonize_or_rexec(int flags, char **argv)
{
	int fd;
	if (flags & DAEMON_CHDIR_ROOT)
		xchdir("/");
	fd = open(bb_dev_null, O_RDWR);
	if (fd < 0) {
		/* NB: we can be called as bb_sanitize_stdio() from init
		 * or mdev, and there /dev/null may legitimately not (yet) exist!
		 * Do not use xopen above, but obtain _ANY_ open descriptor,
		 * even bogus one as below. */
		fd = xopen("/", O_RDONLY); /* don't believe this can fail */
	}
	if (flags & DAEMON_DEVNULL_STDIO) {
		if (flags & DAEMON_DEVNULL_STDIN)
			xdup2(fd, 0);
		if (flags & DAEMON_DEVNULL_OUTERR) {
			xdup2(fd, 1);
			xdup2(fd, 2);
		}
	} else {
		/* have 0,1,2 open at least to /dev/null */
		while ((unsigned)fd < 2)
			fd = dup(fd);
	}
	if (!(flags & DAEMON_ONLY_SANITIZE)) {
		/* fflush_all(); - add it in fork_or_rexec() if necessary */
		if (fork_or_rexec(argv))
			_exit_SUCCESS(); /* parent */
		/* if daemonizing, detach from stdio & ctty */
		setsid();
		dup2(fd, 0);
		dup2(fd, 1);
		dup2(fd, 2);
//		if (flags & DAEMON_DOUBLE_FORK) {
//			/* On Linux, session leader can acquire ctty
//			 * unknowingly, by opening a tty.
//			 * Prevent this: stop being a session leader.
//			 */
//			if (fork_or_rexec(argv))
//				_exit_SUCCESS(); /* parent */
//		}
	}
	while (fd > 2) {
		close(fd--);
		if (!(flags & DAEMON_CLOSE_EXTRA_FDS))
			return;
		/* else close everything after fd#2 */
	}
}
void FAST_FUNC bb_sanitize_stdio(void)
{
	bb_daemonize_or_rexec(DAEMON_ONLY_SANITIZE, NULL);
}
