/* implementation of the LZO1X decompression algorithm
   This file is part of the LZO real-time data compression library.
   Copyright (C) 1996..2008 Markus Franz Xaver Johannes Oberhumer
   All Rights Reserved.
   Markus F.X.J. Oberhumer <markus@oberhumer.com>
   http://www.oberhumer.com/opensource/lzo/
   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.
   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   You should have received a copy of the GNU General Public License
   along with the LZO library; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include "libbb.h"
#include "liblzo.h"
/***********************************************************************
// decompress a block of data.
************************************************************************/
/* safe decompression with overrun testing */
int lzo1x_decompress_safe(const uint8_t* in, unsigned in_len,
		uint8_t* out, unsigned* out_len /*, void* wrkmem */)
{
	register uint8_t* op;
	register const uint8_t* ip;
	register unsigned t;
	register const uint8_t* m_pos = NULL; /* possibly not needed */
	const uint8_t* const ip_end = in + in_len;
	uint8_t* const op_end = out + *out_len;
//	LZO_UNUSED(wrkmem);
	*out_len = 0;
	op = out;
	ip = in;
	if (*ip > 17) {
		t = *ip++ - 17;
		if (t < 4)
			goto match_next;
		assert(t > 0); NEED_OP(t); NEED_IP(t+1);
		do *op++ = *ip++; while (--t > 0);
		goto first_literal_run;
	}
	while (TEST_IP && TEST_OP) {
		t = *ip++;
		if (t >= 16)
			goto match;
		/* a literal run */
		if (t == 0) {
			NEED_IP(1);
			while (*ip == 0) {
				t += 255;
				ip++;
				NEED_IP(1);
			}
			TEST_IV(t);
			t += 15 + *ip++;
		}
		/* copy literals */
		assert(t > 0);
		NEED_OP(t+3);
		NEED_IP(t+4);
		{
			*op++ = *ip++;
			*op++ = *ip++;
			*op++ = *ip++;
			do *op++ = *ip++; while (--t > 0);
		}
 first_literal_run:
		t = *ip++;
		if (t >= 16)
			goto match;
		m_pos = op - (1 + M2_MAX_OFFSET);
		m_pos -= t >> 2;
		m_pos -= *ip++ << 2;
		TEST_LB(m_pos); NEED_OP(3);
		*op++ = *m_pos++;
		*op++ = *m_pos++;
		*op++ = *m_pos;
		goto match_done;
		/* handle matches */
		do {
 match:
			if (t >= 64) {		/* a M2 match */
				m_pos = op - 1;
				m_pos -= (t >> 2) & 7;
				m_pos -= *ip++ << 3;
				t = (t >> 5) - 1;
				TEST_LB(m_pos); assert(t > 0); NEED_OP(t+3-1);
				goto copy_match;
			}
			else if (t >= 32) {			/* a M3 match */
				t &= 31;
				if (t == 0) {
					NEED_IP(1);
					while (*ip == 0) {
						t += 255;
						ip++;
						NEED_IP(1);
					}
					TEST_IV(t);
					t += 31 + *ip++;
				}
				m_pos = op - 1;
				m_pos -= (ip[0] >> 2) + (ip[1] << 6);
				ip += 2;
			}
			else if (t >= 16) {			/* a M4 match */
				m_pos = op;
				m_pos -= (t & 8) << 11;
				t &= 7;
				if (t == 0) {
					NEED_IP(1);
					while (*ip == 0) {
						t += 255;
						ip++;
						NEED_IP(1);
					}
					TEST_IV(t);
					t += 7 + *ip++;
				}
				m_pos -= (ip[0] >> 2) + (ip[1] << 6);
				ip += 2;
				if (m_pos == op)
					goto eof_found;
				m_pos -= 0x4000;
			}
			else {				/* a M1 match */
				m_pos = op - 1;
				m_pos -= t >> 2;
				m_pos -= *ip++ << 2;
				TEST_LB(m_pos); NEED_OP(2);
				*op++ = *m_pos++;
				*op++ = *m_pos;
				goto match_done;
			}
			/* copy match */
			TEST_LB(m_pos); assert(t > 0); NEED_OP(t+3-1);
			{
 copy_match:
				*op++ = *m_pos++; *op++ = *m_pos++;
				do *op++ = *m_pos++; while (--t > 0);
			}
 match_done:
			t = ip[-2] & 3;
			if (t == 0)
				break;
			/* copy literals */
 match_next:
			assert(t > 0);
			assert(t < 4);
			NEED_OP(t);
			NEED_IP(t+1);
			*op++ = *ip++;
			if (t > 1) {
				*op++ = *ip++;
				if (t > 2)
					*op++ = *ip++;
			}
			t = *ip++;
		} while (TEST_IP && TEST_OP);
	}
//#if defined(HAVE_TEST_IP) || defined(HAVE_TEST_OP)
	/* no EOF code was found */
	*out_len = pd(op, out);
	return LZO_E_EOF_NOT_FOUND;
//#endif
 eof_found:
	assert(t == 1);
	*out_len = pd(op, out);
	return (ip == ip_end ? LZO_E_OK :
		   (ip < ip_end	 ? LZO_E_INPUT_NOT_CONSUMED : LZO_E_INPUT_OVERRUN));
//#if defined(HAVE_NEED_IP)
 input_overrun:
	*out_len = pd(op, out);
	return LZO_E_INPUT_OVERRUN;
//#endif
//#if defined(HAVE_NEED_OP)
 output_overrun:
	*out_len = pd(op, out);
	return LZO_E_OUTPUT_OVERRUN;
//#endif
//#if defined(LZO_TEST_OVERRUN_LOOKBEHIND)
 lookbehind_overrun:
	*out_len = pd(op, out);
	return LZO_E_LOOKBEHIND_OVERRUN;
//#endif
}
